#!/bin/bash

EASY_VERSION="1.1"

bb=`tput bold`
nn=`tput sgr0`

# help
if [ $# -eq 0 ]; then
   echo "${bb}-------------------------------------------------------------------${nn}"
   echo "${bb}Creates sof-file for kmo_illumination-recipe and executes it.${nn}"
   echo " "
   echo "- The environment variable ${bb}KMOS_CALIB${nn} pointing to a path has to be"
   echo "  set in order to find the needed static calibration frame WAVE_BAND."
   echo "  Otherwise the execution of this script will fail!"
   echo "- If XCAL, YCAL, LCAL, MASTER_DARK and MASTER_FLAT frames aren't existing"
   echo "  in the working directory the script is searching as well KMOS_CALIB in"
   echo "  order to find them."
   echo "- As argument an arbitrary exposure of an executed "
   echo "  ${bb}KMOS_spec_cal_skyflat${nn} template must be provided."
   echo "  All matching exposures are collected automatically."
   echo "- The first exposure will be ignored when its EXPTIME differs from the"
   echo "  other frames. In this case it is an aquisition frame taken to estimate "
   echo "  the needed EXPTIME."
   echo "- (Optional) If as 2nd argument ${bb}sof${nn} is provided, then only"
   echo "  the sof-file is generated but not executed."
   echo " "
   echo "${bb}Usage:${nn}"
   echo "  easySPARK_illumination.sh <filename with full path>"
   echo "    or"
   echo "  easySPARK_illumination.sh <filename with full path> sof"
   echo " "
   echo "Version: "$EASY_VERSION
   echo "${bb}-------------------------------------------------------------------${nn}"
   exit
fi

# init
sofOnly=0
printVers=0

# extract all arguments beginning with KMOS, eventually imethod and sof
while [ "$1" ] ; do
    tmp=$(basename $1)
    if [[ $tmp == KMOS* ]]; then
        # file-check
        if [ ! -e $1 ]; then
            echo "${bb}ERROR:${nn} File doesn't exist! ($1)"
            exit
        fi
        fileName=$1
    else
        if [[ $tmp == sof ]]; then
            sofOnly=1
        elif [[ $tmp == vers ]]; then
            printVers=1
        fi
    fi
    shift
done

if [ $printVers = 1 ]; then
    echo $EASY_VERSION
    exit
fi

# check if gawk is existing
if command -v gawk >/dev/null; then
   :
else
    echo "'gawk' has not been found on your system!"
    OS=$(uname -s)
    if [ $OS != "Linux" ]; then
        echo "You seem using a MacOS, please use e.g. MacPorts to install gawk."
    fi
    echo "Aborting now.!"
fi

if [ -z "${KMOS_CALIB}" ]; then
   echo "${bb}ERROR:${nn} environment variable KMOS_CALIB is not defined! (see help)"
   exit
fi

#check if file is of correct template
tplId=$(dfits $fileName |\
        fitsort -d tpl.id |\
        gawk '{print $2}')
if [ $tplId != KMOS_spec_cal_skyflat ]; then
   echo "${bb}ERROR${nn}: File is of wrong template: $tplId instead of KMOS_spec_cal_skyflat !"
   exit
fi

# extract TPL.START of ths file
tplStartTime=$(dfits $fileName |\
               fitsort -d tpl.start |\
               gawk '{print $2}')

# get files with matching tplStartTime and FLAT_SKY
dataDir=$(dirname "$fileName")
matchingFiles=$(dfits $dataDir/KMOS*.fits | \
                fitsort -d tpl.start dpr.type |\
       	        grep $tplStartTime |\
                grep FLAT,SKY |\
                gawk '{print $1}' |\
                tr "\n" " ")

# populate sof-file with FLAT_SKY frames
sofFile="illumination_$tplStartTime.sof"
cat > $sofFile </dev/null
for file in $matchingFiles; do
    echo "$file                FLAT_SKY" >> $sofFile
done

# get grating id of first FLAT_SKY frame
gratingUpper=$(dfits $matchingFiles |\
               fitsort -d ins.grat1.id |\
               head -n1 |\
               gawk '{print $2}')

# add static calibration frames
echo "$KMOS_CALIB/kmos_wave_band.fits       WAVE_BAND" >> $sofFile

# add band-specific calibration frames
fnMasterDark=master_dark.fits
fnMasterFlat=master_flat_$gratingUpper$gratingUpper$gratingUpper.fits
fnXcal=xcal_$gratingUpper$gratingUpper$gratingUpper.fits
fnYcal=ycal_$gratingUpper$gratingUpper$gratingUpper.fits
fnLcal=lcal_$gratingUpper$gratingUpper$gratingUpper.fits
fnFlatEdge=flat_edge_$gratingUpper$gratingUpper$gratingUpper.fits

if [ ! -e $fnMasterDark ]; then
   fnMasterDark=$KMOS_CALIB"/"$fnMasterDark
fi
if [ ! -e $fnMasterFlat ]; then
   fnMasterFlat=$KMOS_CALIB"/"$fnMasterFlat
fi
if [ ! -e $fnXcal ]; then
   fnXcal=$KMOS_CALIB"/"$fnXcal
fi
if [ ! -e $fnYcal ]; then
   fnYcal=$KMOS_CALIB"/"$fnYcal
fi
if [ ! -e $fnLcal ]; then
   fnLcal=$KMOS_CALIB"/"$fnLcal
fi
if [ ! -e $fnFlatEdge ]; then
   fnFlatEdge=$KMOS_CALIB"/"$fnFlatEdge
fi

echo "$fnMasterDark      MASTER_DARK" >> $sofFile
echo "$fnMasterFlat      MASTER_FLAT" >> $sofFile
echo "$fnXcal            XCAL" >> $sofFile
echo "$fnYcal            YCAL" >> $sofFile
echo "$fnLcal            LCAL" >> $sofFile
echo "$fnFlatEdge        FLAT_EDGE" >> $sofFile

echo ""
echo "${bb}------------------ easySPARK_illumination ---------------------${nn}"
echo "${bb}INPUT${nn}"
echo "filename:  "$fileName
echo "with "
echo "TPL.START: "$tplStartTime
echo "GRATING:   "$gratingUpper
echo "EXPTIME:   "$exptime
echo ""
echo "${bb}OUTPUT${nn}"
echo $sofFile" with:"
cat $sofFile
echo ""

# execute kmo_illumination
if [ $sofOnly = 0 ]; then
   echo "${bb}NOW CALCULATING ILLUMINATION CORRECTION:${nn}"
   echo "esorex kmo_illumination "$sofFile
   
   esorex kmo_illumination $sofFile
fi
echo "${bb}----------------------------------------------------------${nn}"
